/*
 * main.c
 *
 * ============================================================================
 * Copyright (c) Texas Instruments Inc 2005
 *
 * Use of this software is controlled by the terms and conditions found in the
 * license agreement under which this software has been supplied or provided.
 * ============================================================================
 */

/* Standard Linux headers */
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <getopt.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Codec Engine headers */
#include <xdc/std.h>
#include <ti/sdo/ce/Engine.h>
#include <ti/sdo/ce/trace/gt.h>
#include <ti/sdo/ce/CERuntime.h>
#include <ti/sdo/ce/utils/trace/TraceUtil.h>

/* Demo headers */
#include <rendezvous.h>
#include "decode.h"
#include "display.h"
#include "video.h"

#define FBIO_GETSTD         _IOR('F', 0x25, u_int32_t)

/* The levels of initialization */
#define LOGSINITIALIZED     0x1
#define RENDEZVOUSOPENED    0x2
#define VIDEOTHREADCREATED  0x4

typedef struct Args {
    VideoDecoder   videoDecoder;
    char          *videoFile;
    Resolution     resolution;
    int            loop;
    int            time;
} Args;

/* Global variable declarations for this application */
GlobalData gbl = { 0, 1, 0, 0, 0, 0, 0, FALSE, NOSTD };

/******************************************************************************
 * usage
 ******************************************************************************/
static void usage(void)
{
    printf("Usage: decode [options]\n\n"
           "Options:\n"
           "-v | --videofile   Video file to play\n"
           "-t | --time        Number of seconds to run the demo [infinite]\n"
	   "-r | --resolution  ntsc pal vga 720p 1080i\n"
           "-l | --loop        Loop to beginning of files when done [off]\n"
           "-h | --help        Print this message\n\n"
           "You must supply a video file and specify resolution\n"
           "with appropriate extensions for the file formats.\n");
}

/******************************************************************************
 * parseArgs
 ******************************************************************************/
static void parseArgs(int argc, char *argv[], Args *argsp)
{
    const char shortOptions[] = "v:t:r:lh";
    const struct option longOptions[] = {
        {"videofile", required_argument, NULL, 'v'},
        {"time", required_argument, NULL, 't'},
	{"resolution", required_argument, NULL, 'r'},
        {"loop", no_argument, NULL, 'l'},
        {"help", no_argument, NULL, 'h'},
        {0, 0, 0, 0}
    };
    int     index;
    int     c;
    char    *extension;

    for (;;) {
        c = getopt_long(argc, argv, shortOptions, longOptions, &index);

        if (c == -1) {
            break;
        }

        switch (c) {
            case 0:
                break;

            case 'v':
                extension = rindex(optarg, '.');
                if (extension == NULL) {
                    fprintf(stderr, "Video file without extension: %s\n",
                            optarg);
                    break;
                }
                if (strcmp(extension, ".m2v") == 0) {
                    argsp->videoDecoder = MPEG2_VIDEO_DECODER;
                }
                else if (strcmp(extension, ".264") == 0) {
                    argsp->videoDecoder = H264_VIDEO_DECODER;
                }
                else if (strcmp(extension, ".mpeg4") == 0) {
                    argsp->videoDecoder = MPEG4_VIDEO_DECODER;
                }
                else {
                    fprintf(stderr, "Unknown video file extension: %s\n",
                            extension);
                    exit(EXIT_FAILURE);
                }
                argsp->videoFile = optarg;

                break;

            case 't':
                argsp->time = atoi(optarg);
                break;

	    case 'r':
                if (strcmp(optarg, "ntsc") == 0) {
                    argsp->resolution = RES_NTSC;
                }
                else if (strcmp(optarg, "pal") == 0) {
		    argsp->resolution = RES_PAL;
                }
                else if (strcmp(optarg, "vga") == 0) {
		    argsp->resolution = RES_VGA;
                }
                else if (strcmp(optarg, "720p") == 0) {
                    argsp->resolution = RES_720P;
                }
                else if (strcmp(optarg, "1080i") == 0) {
                    argsp->resolution = RES_1080I;
                }
                else {
                    fprintf(stderr, "Unknown Resolution: %s\n",
                            optarg);
                    exit(EXIT_FAILURE);
                }

                break;
	        
            case 'l':
                argsp->loop = TRUE;
                break;

            case 'h':
                usage();
                exit(EXIT_SUCCESS);

            default:
                usage();
                exit(EXIT_FAILURE);
        }
    }

    /* Need a video file to decode and resolution specification */
    if (!argsp->resolution || !argsp->videoDecoder) {
      usage();
      exit(EXIT_FAILURE);
    }
}



/******************************************************************************
 * main
 ******************************************************************************/
int main(int argc, char *argv[])
{
    unsigned int       initMask  = 0;
    int                status    = EXIT_SUCCESS;
    int                numThreads;
    struct sched_param schedParam;
    Rendezvous_Obj     rendezvous;
    pthread_t          videoThread;
    VideoEnv           videoEnv;
    pthread_attr_t     attr;
    void              *ret;
    Args               args = {
        0,
        NULL,
	0,
        0,
        0
    }; 

    /* Parse the arguments given to the app and set the app environment */
    parseArgs(argc, argv, &args);

    printf("Decode demo started.\n");

    /* Initialize the mutex which protects the global data */
    pthread_mutex_init(&gbl.mutex, NULL);

    /* Set the priority of this whole process to max (requires root) */
    setpriority(PRIO_PROCESS, 0, -20);

    /* Initialize Codec Engine runtime */
    CERuntime_init();

    DBG("Codec Engine initialized\n");

    /* Initialize the logs. Must be done after CERuntime_init() */
    TraceUtil_start(ENGINE_NAME);

    initMask |= LOGSINITIALIZED;

    DBG("Logging initialized\n");

    /* Open the object which synchronizes the thread initialization */
    numThreads = 2;
    Rendezvous_open(&rendezvous, numThreads);
    
    initMask |= RENDEZVOUSOPENED;

    DBG("Rendezvous opened for %d threads\n", numThreads);

    /* Initialize the thread attributes */
    if (pthread_attr_init(&attr)) {
        ERR("Failed to initialize thread attrs\n");
        cleanup(EXIT_FAILURE);
    }

    /* Force the thread to use custom scheduling attributes */
    if (pthread_attr_setinheritsched(&attr, PTHREAD_EXPLICIT_SCHED)) {
        ERR("Failed to set schedule inheritance attribute\n");
        cleanup(EXIT_FAILURE);
    }

    /* Set the thread to be fifo real time scheduled */
    if (pthread_attr_setschedpolicy(&attr, SCHED_FIFO)) {
        ERR("Failed to set FIFO scheduling policy\n");
        cleanup(EXIT_FAILURE);
    }
    
    /* Start the video thread if a file name is supplied */
    if (args.videoFile) {
        /* Set the thread priority */
        schedParam.sched_priority = sched_get_priority_max(SCHED_FIFO)-1;
        if (pthread_attr_setschedparam(&attr, &schedParam)) {
            ERR("Failed to set scheduler parameters\n");
            cleanup(EXIT_FAILURE);
        }

        /* Create the thread */
        videoEnv.hRendezvous  = &rendezvous;
        videoEnv.videoFile    = args.videoFile;
        videoEnv.videoDecoder = args.videoDecoder;
        videoEnv.loop         = args.loop;
	videoEnv.resolution   = args.resolution;

        if (pthread_create(&videoThread, &attr, videoThrFxn, &videoEnv)) {
            ERR("Failed to create video thread\n");
            cleanup(EXIT_FAILURE);
        }

        initMask |= VIDEOTHREADCREATED;

	/*	ret = videoThrFxn(&videoEnv);

	if (ret == THREAD_FAILURE) {
	  status = EXIT_FAILURE;
	}

        initMask |= VIDEOTHREADCREATED;*/
    }    

    while(1)
      usleep(300000);

cleanup:
    /* Make sure the other threads aren't waiting for init to complete */
    Rendezvous_force(&rendezvous);

    if (initMask & VIDEOTHREADCREATED) {
        if (pthread_join(videoThread, &ret) == 0) {
            if (ret == THREAD_FAILURE) {
                status = EXIT_FAILURE;
            }
        }
    }

    if (initMask & RENDEZVOUSOPENED) {
        Rendezvous_close(&rendezvous);
    }

    if (initMask & LOGSINITIALIZED) {
        TraceUtil_stop();
    }

    /* Destroy the global mutex */
    pthread_mutex_destroy(&gbl.mutex);

    exit(status);
}
