#include <msp430.h> 
#include <stdio.h>
#include <math.h>


//Defines output voltages for virtual power tree
//Array entries numbers correspond directly to vout #s
const float outputVoltages[] = {1700,1700,1700,1700,1700,1700,3000,2500,2500,2500,2500,2500};


//Determines if each virtual power rail requires an enable signal to start up
//0 = EN not required
//1 = EN required
const int requiresEN = 1;


void setClock24MHz()
{
    // Configure two FRAM waitstate as required by the device datasheet for MCLK
    // operation at 24MHz(beyond 8MHz) _before_ configuring the clock system.
    FRCTL0 = FRCTLPW | NWAITS_2;

    __bis_SR_register(SCG0);                           // disable FLL
    CSCTL3 |= SELREF__REFOCLK;                         // Set REFO as FLL reference source
    CSCTL0 = 0;                                        // clear DCO and MOD registers
    CSCTL1 |= DCORSEL_7;                               // Set DCO = 24MHz
    CSCTL2 = FLLD_0 + 731;                             // DCOCLKDIV = 24MHz
    __delay_cycles(3);
    __bic_SR_register(SCG0);                           // enable FLL
    while(CSCTL7 & (FLLUNLOCK0 | FLLUNLOCK1));         // FLL locked

    CSCTL4 = SELMS__DCOCLKDIV | SELA__REFOCLK;        // set default REFO(~32768Hz) as ACLK source, ACLK = 32768Hz
                                                       // default DCOCLKDIV as MCLK and SMCLK source

    PM5CTL0 &= ~LOCKLPM5;                              // Disable the GPIO power-on default high-impedance mode
                                                       // to activate previously configured port settings
}

void configurePowerTree()
{
    ///////////////////////////////////////////////////////////////////////////////////////////////
    // PWM outputs
    ///////////////////////////////////////////////////////////////////////////////////////////////

    // Setup TimerB0 (P1.6 and P1.7 Outputs)
    TB0CCR0 = 330;                                    // PWM Period
    TB0CCTL1 = OUTMOD_7;                              // CCR1 reset/set
    TB0CCTL2 = OUTMOD_7;                              // CCR2 reset/set
    TB0CTL = TBSSEL_2 | MC_1 | TBCLR;                 // SMCLK, up mode, clear TBR

    // Setup TimerB1 (P2.0 and P2.1 Outputs)
    TB1CCR0 = 330;                                //PWM Period
    TB1CCTL1 = OUTMOD_7;                          // CCR1 reset/set
    TB1CCTL2 = OUTMOD_7;                              // CCR2 reset/set
    TB1CTL = TBSSEL_2 | MC_1 | TBCLR;                 // SMCLK, up mode, clear TBR

    // Setup TimerB3 (P6.0, P6.1, P6.2 and P6.3 Outputs)
    TB3CCR0 = 330;                                //PWM Period
    TB3CCTL1 = OUTMOD_7;                          // CCR1 reset/set
    TB3CCTL2 = OUTMOD_7;                              // CCR2 reset/set
    TB3CCTL3 = OUTMOD_7;                              // CCR2 reset/set
    TB3CCTL4 = OUTMOD_7;                              // CCR2 reset/set
    TB3CTL = TBSSEL_2 | MC_1 | TBCLR;                 // SMCLK, up mode, clear TBR

    if(requiresEN == 0)
    {
        TB0CCR1 = outputVoltages[4]/10; //set pwm duty cycles
        TB0CCR2 = outputVoltages[5]/10;
        TB1CCR1 = outputVoltages[6]/10;
        TB1CCR2 = outputVoltages[7]/10;
        TB3CCR1 = outputVoltages[8]/10;
        TB3CCR2 = outputVoltages[9]/10;
        TB3CCR3 = outputVoltages[10]/10;
        TB3CCR4 = outputVoltages[11]/10;
    }
    else
    {
        TB0CCR1 = 0;
        TB0CCR2 = 0;
        TB1CCR1 = 0;
        TB1CCR2 = 0;
        TB3CCR1 = 0;
        TB3CCR2 = 0;
        TB3CCR3 = 0;
        TB3CCR4 = 0;
    }

    //configure PWM output pins (P1.6 and P1.7 Outputs)
    P1DIR |= BIT6 | BIT7;
    P1SEL1 |= BIT6 | BIT7;
    //configure PWM output pins (P2.0 and P2.1 Outputs)
    P2DIR |= BIT0 | BIT1;
    P2SEL0 |= BIT0 | BIT1;
    //configure PWM output pins (P6.0, P6.1, P6.2 and P6.3 Outputs)
    P6DIR |= BIT0 | BIT1 | BIT2 | BIT3;
    P6SEL0 |= BIT0 | BIT1 | BIT2 | BIT3;


    ///////////////////////////////////////////////////////////////////////////////////////////////
    // DAC outputs
    ///////////////////////////////////////////////////////////////////////////////////////////////
    P1SEL0 |= BIT1 | BIT5;                           // Select P1.1 as OA0O function
    P1SEL1 |= BIT1 | BIT5;                           // OA is used as buffer for DAC
    P3SEL0 |= BIT1 | BIT5;
    P3SEL1 |= BIT1 | BIT5;

    PM5CTL0 &= ~LOCKLPM5;                     // Disable the GPIO power-on default high-impedance mode
                                              // to activate previously configured port settings


    // Configure internal vref
    PMMCTL0_H = PMMPW_H;                    // Unlock the PMM registers
    PMMCTL2 = INTREFEN | REFVSEL_2;           // Enable internal 2.5V reference
    while(!(PMMCTL2 & REFGENRDY));            // Poll till internal reference settles



    // Enable DAC0
    SAC0DAC = DACSREF_1;                      // Select int Vref as DAC reference
    SAC0DAC |= DACEN;                         // Enable DAC
    // Enable DAC1
    SAC1DAC = DACSREF_1;                      // Select int Vref as DAC reference
    SAC1DAC |= DACEN;                         // Enable DAC
    // Enable DAC2
    SAC2DAC = DACSREF_1;                      // Select int Vref as DAC reference
    SAC2DAC |= DACEN;                         // Enable DAC
    // Enable DAC3
    SAC3DAC = DACSREF_1;                      // Select int Vref as DAC reference
    SAC3DAC |= DACEN;                         // Enable DAC

    // Configure SAC0
    SAC0OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_1; // Select positive and negative pin input
    SAC0OA |= OAPM;                             // Select low speed and low power mode
    SAC0PGA = MSEL_1;                           // Set OA as buffer mode
    SAC0OA |= SACEN + OAEN;                     // Enable SAC and OA
    // Configure SAC1
    SAC1OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_1; // Select positive and negative pin input
    SAC1OA |= OAPM;                             // Select low speed and low power mode
    SAC1PGA = MSEL_1;                           // Set OA as buffer mode
    SAC1OA |= SACEN + OAEN;                     // Enable SAC and OA
    // Configure SAC2
    SAC2OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_1; // Select positive and negative pin input
    SAC2OA |= OAPM;                             // Select low speed and low power mode
    SAC2PGA = MSEL_1;                           // Set OA as buffer mode
    SAC2OA |= SACEN + OAEN;                     // Enable SAC and OA
    // Configure SAC3
    SAC3OA = NMUXEN + PMUXEN + PSEL_1 + NSEL_1; // Select positive and negative pin input
    SAC3OA |= OAPM;                             // Select low speed and low power mode
    SAC3PGA = MSEL_1;                           // Set OA as buffer mode
    SAC3OA |= SACEN + OAEN;                     // Enable SAC and OA



    //Configure DAC output pins
    P1SEL0 |= BIT1 | BIT5;                           // Select P1.1 as OA0O function
    P1SEL1 |= BIT1 | BIT5;                           // OA is used as buffer for DAC
    P3SEL0 |= BIT1 | BIT5;
    P3SEL1 |= BIT1 | BIT5;

    // Set output voltages
   if(requiresEN == 0)
    {
       //Op amp circuits have 3.4X gain. divide by 8600 given a 5V power supply 5400?
       SAC0DAT = (outputVoltages[0] / 8600)*4095;     // Initial DAC0 data
       SAC1DAT = (outputVoltages[1] / 8600)*4095;     // Initial DAC1 data
       SAC2DAT = (outputVoltages[2] / 8600)*4095;     // Initial DAC2 data
       SAC3DAT = (outputVoltages[3] / 8600)*4095;     // Initial DAC3 data


    }
    else
    {
       SAC0DAT = 0;     // Initial DAC0 data
       SAC1DAT = 0;     // Initial DAC1 data
       SAC2DAT = 0;     // Initial DAC2 data
       SAC3DAT = 0;     // Initial DAC3 data
    }
}

void configureENPins()
{
    //set enable pins to input mode
    P6DIR &= ~(BIT4);
    P3DIR &= ~(BIT7 | BIT3 | BIT2 | BIT0);
    P2DIR &= ~(BIT5 | BIT4 | BIT2);
    P4DIR &= ~(BIT7 | BIT6 | BIT4 | BIT1 | BIT0);

    P6SEL0 &= ~(BIT4);
    P3SEL0 &= ~(BIT7 | BIT3 | BIT2 | BIT0);
    P2SEL0 &= ~(BIT5 | BIT4 | BIT2);
    P4SEL0 &= ~(BIT7 | BIT6 | BIT4 | BIT1 | BIT0);

    P6SEL1 &= ~(BIT4);
    P3SEL1 &= ~(BIT7 | BIT3 | BIT2 | BIT0);
    P2SEL1 &= ~(BIT5 | BIT4 | BIT2);
    P4SEL1 &= ~(BIT7 | BIT6 | BIT4 |BIT1 | BIT0);

    //set pullup config
    P6REN |= (BIT4);
    P3REN |= (BIT7 | BIT3 | BIT2 | BIT0);
    P2REN |= (BIT5 | BIT4 | BIT2);
    P4REN |= (BIT7 | BIT6 | BIT4 | BIT1 | BIT0);

    P6OUT |= (BIT4);
    P3OUT |= (BIT7 | BIT3 | BIT2 | BIT0);
    P2OUT |= (BIT5 | BIT4 | BIT2);
    P4OUT |= (BIT7 | BIT6 | BIT4 | BIT1 | BIT0);
}

void configureLEDPins()
{
    P5DIR |= (BIT4 | BIT3 | BIT1 | BIT0); //set pins to output
    P5SEL0 &= ~(BIT4 | BIT3 | BIT1 | BIT0);
    P5SEL1 &= ~(BIT4 | BIT3 | BIT1 | BIT0);

    P5OUT &= ~(BIT4 | BIT3 | BIT1 | BIT0); //set pin outputs to 0
}

void configureGLEVPins()
{
    P5DIR &= ~(BIT2); //set NIRQ pin to input
    P3DIR &= ~(BIT6); //set RST pin to input
    P5SEL0 &= ~(BIT2);
    P5SEL1 &= ~(BIT2);
    P3SEL0 &= ~(BIT6);
    P3SEL1 &= ~(BIT6);

    P5REN &= ~(BIT2); //no pullup or pulldown
    P3REN &= ~(BIT6); //no pullup or pulldown

    //configure ACT pins
    P4DIR |= (BIT5); //Greenland ACT
    P3DIR |= (BIT4); //Everest ACT
    P4SEL0 &= ~(BIT5);
    P4SEL1 &= ~(BIT5);
    P3SEL0 &= ~(BIT4);
    P3SEL1 &= ~(BIT4);

    P4OUT &= ~(BIT5); //set outputs to 0
    P3OUT &= ~(BIT4);

}


/**
 * main.c
 */
int main(void)
{

    WDTCTL = WDTPW | WDTHOLD;	// stop watchdog timer


	setClock24MHz();
	configurePowerTree();
	configureENPins();
	configureLEDPins();
	configureGLEVPins();


    //wait for voltages to settle before enabling ACT
     __delay_cycles(100000000); //wait ~2000ms

	//enable ACT pins
	P3OUT |= (BIT4); //Enable Everest ACT
	P4OUT |= (BIT5); //Enable Greenland ACT


	while(1)
    {


        if(requiresEN == 1)
        {
            //EN1
            if((P2IN & BIT2) != 0)
            {
                SAC0DAT = (outputVoltages[0] / 8600)*4095; //Set DAC output to user inputed value
            }
            else
            {
                SAC0DAT = 0; //Set DAC output to 0
            }
            //EN2
            if((P4IN & BIT0) != 0)
            {
                SAC1DAT = (outputVoltages[1] / 8600)*4095; //Set DAC output to user inputed value
            }
            else
            {
                SAC1DAT = 0; //Set DAC output to 0
            }
            //EN3
            if((P4IN & BIT6) != 0)
            {
                SAC2DAT = (outputVoltages[2] / 8600)*4095; //Set DAC output to user inputed value
            }
            else
            {
                SAC2DAT = 0; //Set DAC output to 0
            }
            //EN4
            if((P4IN & BIT7) != 0)
            {
                SAC3DAT = (outputVoltages[3] / 8600)*4095; //Set DAC output to user inputed value
            }
            else
            {
                SAC3DAT = 0; //Set DAC output to 0
            }
            //EN5
            if((P4IN & BIT4) != 0)
            {
                TB0CCR1 = outputVoltages[4]/10;
            }
            else
            {
                TB0CCR1 = 0;
            }
            //EN6
            if((P2IN & BIT5) != 0)
            {
                TB0CCR2 = outputVoltages[5]/10;
            }
            else
            {
                TB0CCR2 = 0;
            }
            //EN7
            if((P3IN & BIT0) != 0)
            {
                TB1CCR1 = outputVoltages[6]/10;
            }
            else
            {
                TB1CCR1 = 0;
            }
            //EN8
            if((P6IN & BIT4) != 0)
            {
                TB1CCR2 = outputVoltages[7]/10;
            }
            else
            {
                TB1CCR2 = 0;
            }
            //EN9
            if((P3IN & BIT7) != 0)
            {
                TB3CCR1 = outputVoltages[8]/10;
            }
            else
            {
                TB3CCR1 = 0;
            }
            //EN10
            if((P2IN & BIT4) != 0)
            {
                TB3CCR2 = outputVoltages[9]/10;
            }
            else
            {
                TB3CCR2 = 0;
            }
            //EN11
            if((P3IN & BIT3) != 0)
            {
                TB3CCR3 = outputVoltages[10]/10;
            }
            else
            {
                TB3CCR3 = 0;
            }
            //EN12
            if((P3IN & BIT2) != 0)
            {
                TB3CCR4 = outputVoltages[11]/10;
            }
            else
            {
                TB3CCR4 = 0;
            }

            //LED routine
            //D1 NIRQ
            if((P5IN & BIT2) == 0)
            {
                P5OUT |= BIT3;
            }
            else
            {
                P5OUT &= ~(BIT3);
            }
            //D2 EN1
            if((P2IN & BIT2) != 0)
            {
                P5OUT |= BIT1;
            }
            else
            {
                P5OUT &= ~(BIT1);
            }
            //D3 EN2
            if((P4IN & BIT0) != 0)
            {
                P5OUT |= BIT0;
            }
            else
            {
                P5OUT &= ~(BIT0);
            }
            //D4 EN3
            if((P4IN & BIT6) != 0)
            {
                P5OUT |= (BIT4);
            }
            else
            {
                P5OUT &= ~(BIT4);
            }
        }
        else //LEDs still work with no enable
        {
           //LED routine
           //D1 NIRQ
           if((P5IN & BIT2) == 0)
           {
               P5OUT |= BIT3;
           }
           else
           {
               P5OUT &= ~(BIT3);
           }
           //D2 EN1
           if((P2IN & BIT2) != 0)
           {
               P5OUT |= BIT1;
           }
           else
           {
               P5OUT &= ~(BIT1);
           }
           //D3 EN2
           if((P4IN & BIT0) != 0)
           {
               P5OUT |= BIT0;
           }
           else
           {
               P5OUT &= ~(BIT0);
           }
           //D4 EN3
           if((P4IN & BIT6) != 0)
           {
               P5OUT |= (BIT4);
           }
           else
           {
               P5OUT &= ~(BIT4);
           }
        }

    if((P4IN & BIT1) == 0)
	{
	    P3OUT &= ~(BIT4); //Enable Everest ACT
	    P4OUT &= ~(BIT5); //Enable Greenland ACT
	}
	else
	{
	    P3OUT |= (BIT4); //Enable Everest ACT
	    P4OUT |= (BIT5); //Enable Greenland ACT
	}
}
	return 0;
}
