/* --COPYRIGHT--,BSD
 * Copyright (c) 2019, Texas Instruments Incorporated
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * *  Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * *  Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * *  Neither the name of Texas Instruments Incorporated nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * --/COPYRIGHT--*/
//*****************************************************************************
//         Dual Ray Smoke Sensor Software Library
//
// This library implements functions to read TI smoke sensors
//
// Texas Instruments, Inc.
// Jan 2020
// Luis R.

#ifndef __DUALRAYSMOKEAFELIB_H__
#define __DUALRAYSMOKEAFELIB_H__

#include <msp430.h>
#include <stdint.h>
#include <stdbool.h>
#include <IQmathLib.h>
#include <DualRaySmokeAFELib_Config.h>
#include <DualRaySmokeAFE_HAL.h>


//! \brief Configuration of SW Library
typedef struct
{
    //!< Indicates if alarm detection algorithm is enabled or disabled.
    //!     Set by application dynamically if needed.
    bool bAlarmDetectionEnabled;
    //!< Reflection delta for each LED before triggering an alarm
    //!     Set by application dynamically if needed, or as a default.
    int16_t i16RefDelta_Threshold[DUALRAYSMOKEAFE_LED_TOTAL];
} tDualRaySmokeAFELib_Config;

//! \brief Contains measurement information per LED
typedef struct
{
    //!< Dark measurement
    //!     Calculated by algorithm based on measurement.
    uint16_t u16Dark;
    //!< Light measurement
    //!     Calculated by algorithm based on measurement.
    uint16_t u16Light;
    //!< Difference between Light and Dark
    //!     Calculated by algorithm based on measurement.
    int16_t  i16Diff;
} tDualRaySmokeAFELib_MeasData;

//! \brief Main structure with Data for Dual Ray Demo
typedef struct
{
    //!< LED data with measurements.
    //!     Calculated and updated by algorithm based on measurement.
    tDualRaySmokeAFELib_MeasData sLED[DUALRAYSMOKEAFE_LED_TOTAL];
    //!< Temperature measurement in IQ15 format (17 integer, 15 fractional)
    _iq15 iq15TemperatureC;
} tDualRaySmokeAFELib_DualRayData;

//! \brief Alarm state machine
typedef enum
{
    DUALRAYSMOKEAFE_ALARM_OFF = 0,
    DUALRAYSMOKEAFE_ALARM_WARNING1,
    DUALRAYSMOKEAFE_ALARM_WARNING2,
    DUALRAYSMOKEAFE_ALARM_ON,
    DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDHIGH,
    DUALRAYSMOKEAFE_ALARM_ON_THRESHOLDLOW,
    DUALRAYSMOKEAFE_ALARM_TURN_ON,
    DUALRAYSMOKEAFE_ALARM_TURN_OFF,
    DUALRAYSMOKEAFE_ALARM_SILENT,
}tDualRaySmokeAFE_AlarmDetectionState;

//************* Library and Detection Algorithm ********************************
//! \brief Initializes the AFE Library and Detection Algorithm.
//!     This function initializes both the library and the corresponding HAL
//!     functions for smoke detector AFE.
//!
//! \param[in] LibConfig is the pointer to the library configuration structure.
//! \param[in] HALConfig is the pointer to the HAL configuration structure.
//!
//! \return none
extern void DualRaySmokeAFELib_Init(tDualRaySmokeAFELib_Config *LibConfig,
                                    tDualRaySmokeAFE_HAL_Config *HALConfig);

//! \brief Updates the configuration of library.
//!     This function should be called if parameters are modified dynamically
//!     by the application.
//!
//! \param[in] LibConfig is the pointer to the library configuration structure.
//! \param[in] HALConfig is the pointer to the HAL configuration structure.
//!
//! \return none
extern void DualRaySmokeAFELib_UpdateConfig(tDualRaySmokeAFELib_Config *LibConfig,
                                            tDualRaySmokeAFE_HAL_Config *HALConfig);

//! \brief Runs basic algorithm to check if light was detected for either
//!     LED and decides if the alarm should be turned on or off.
//!
//! \param[in] Data is the pointer to the library data structure.
//!
//! \return tDualRaySmokeAFE_AlarmDetectionState is the new state of alarm
//!         state machine.
extern tDualRaySmokeAFE_AlarmDetectionState DualRaySmokeAFELib_DetectionAlgorithm(
                                          tDualRaySmokeAFELib_DualRayData *Data);

//! \brief Returns the state of the alarm state machine.
//!
//! \return tDualRaySmokeAFE_AlarmDetectionState is the current state of
//!         alarm state machine.
extern tDualRaySmokeAFE_AlarmDetectionState DualRaySmokeAFELib_AlarmGetState(void);

//! \brief Sets the state of the alarm state machine to Silent
//!         where the alarm won't sound for DUALRAYSMOKEAFELIB_ALARM_SILENT_SEC
//!         seconds.
//!
//! \return none
extern void DualRaySmokeAFELib_setAlarmSilent(void);

//! \brief Turn ON the alarm.
//!         Updates the state machine and controls the alarm HW.
//!
//! \param[in] sounderOn enables sounder when true.
//!
//! \return none
extern void DualRaySmokeAFELib_setAlarmOn(bool sounderOn);

//! \brief Turn OFF the alarm.
//!         Updates the state machine and controls the alarm HW.
//!
//! \return none
extern void DualRaySmokeAFELib_setAlarmOff(void);

//************* Measurement ***************************************************
//! \brief Perform a measurement of both LEDs
//!
//! \param[out] SmokeData is the pointer to the data structure which will be
//!                 updated with LED information.
//! \return none
extern void DualRaySmokeAFELib_Measurement(
                                tDualRaySmokeAFELib_DualRayData *SmokeData);

//! \brief Perform a temperature measurement
//!
//! \param[out] SmokeData is the pointer to the data structure which will be
//!                 updated with temperature information.
//! \return none
extern void DualRaySmokeAFELib_TemperatureMeasurement(
                                tDualRaySmokeAFELib_DualRayData *SmokeData);


#endif //__DUALRAYSMOKEAFELIB_H__
